<?php
session_start();
if (!isset($_SESSION['admin_logged_in'])) {
    header("Location: admin_login.php");
    exit();
}

require_once 'config.php';
$conn = getDBConnection();
$message = '';

if ($_SERVER['REQUEST_METHOD'] == 'POST' && isset($_POST['create_initial_knockout'])) {
    $pairings = $_POST['pairings'] ?? [];
    
    $created_count = 0;
    foreach ($pairings as $pairing) {
        if (!empty($pairing['team1']) && !empty($pairing['team2'])) {
            $team1_id = intval($pairing['team1']);
            $team2_id = intval($pairing['team2']);
            
            if ($team1_id == $team2_id) {
                continue;
            }
            
            $referee_id = !empty($pairing['referee']) ? intval($pairing['referee']) : null;
            $court_name = !empty($pairing['court']) ? trim($pairing['court']) : null;
            $start_time = !empty($pairing['time']) ? trim($pairing['time']) : null;
            
            $stmt = $conn->prepare("INSERT INTO matches (pool_id, round_number, team1_id, team2_id, referee_id, court_name, start_time, match_type, status) VALUES (1, 1, ?, ?, ?, ?, ?, 'knockout', 'not_started')");
            $stmt->bind_param("iiiss", $team1_id, $team2_id, $referee_id, $court_name, $start_time);
            
            if ($stmt->execute()) {
                $created_count++;
            }
            $stmt->close();
        }
    }
    
    $message = "$created_count knockout matches created successfully! Future rounds will auto-advance after completion.";
}

$all_teams = [];
$teams_query = "SELECT t.*, p.pool_number 
                FROM teams t 
                JOIN pools p ON t.pool_id = p.id 
                ORDER BY p.pool_number, t.name";
$teams_result = $conn->query($teams_query);

while ($team = $teams_result->fetch_assoc()) {
    $all_teams[] = [
        'id' => $team['id'],
        'name' => $team['name'],
        'pool_number' => $team['pool_number']
    ];
}

$referees_result = $conn->query("SELECT * FROM referees ORDER BY name");

$knockout_count_query = "SELECT COUNT(*) as count FROM matches WHERE match_type = 'knockout'";
$knockout_count = $conn->query($knockout_count_query)->fetch_assoc()['count'];

function getRoundNameByMatches($matches_in_round) {
    $teams_in_round = $matches_in_round * 2;
    
    if ($teams_in_round == 2) return "Finals";
    if ($teams_in_round == 4) return "Semi-Finals";
    if ($teams_in_round == 8) return "Quarter-Finals";
    if ($teams_in_round == 16) return "Round of 16";
    if ($teams_in_round == 32) return "Round of 32";
    if ($teams_in_round == 64) return "Round of 64";
    return "Round of " . $teams_in_round;
}

$existing_knockout = $conn->query("SELECT m.*, t1.name as team1_name, t2.name as team2_name, r.name as referee_name 
                                  FROM matches m 
                                  JOIN teams t1 ON m.team1_id = t1.id 
                                  JOIN teams t2 ON m.team2_id = t2.id 
                                  LEFT JOIN referees r ON m.referee_id = r.id 
                                  WHERE m.match_type = 'knockout' 
                                  ORDER BY m.round_number, m.id");

$rounds_display = [];
while ($match = $existing_knockout->fetch_assoc()) {
    $rounds_display[$match['round_number']][] = $match;
}
?>
<!DOCTYPE html>
<html lang="en">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>Knockout Setup</title>
    <link rel="stylesheet" href="style.css">
    <script>
        let usedTeams = [];
        
        function updateTeamDropdowns() {
            const selects = document.querySelectorAll('select[name*="[team"]');
            usedTeams = [];
            
            selects.forEach(select => {
                if (select.value) {
                    usedTeams.push(parseInt(select.value));
                }
            });
            
            selects.forEach(select => {
                const currentValue = select.value;
                const options = select.querySelectorAll('option');
                
                options.forEach(option => {
                    if (option.value === '') return; 
                    
                    const teamId = parseInt(option.value);
                    if (usedTeams.includes(teamId) && teamId != currentValue) {
                        option.disabled = true;
                        option.style.color = '#ccc';
                    } else {
                        option.disabled = false;
                        option.style.color = '';
                    }
                });
            });
        }
        
        function calculateMatches() {
            const numTeams = parseInt(document.getElementById('num_teams').value);
            const container = document.getElementById('matchesContainer');
            const numMatches = numTeams / 2;
            
            container.innerHTML = '';
            
            for (let i = 0; i < numMatches; i++) {
                const matchDiv = document.createElement('div');
                matchDiv.style.cssText = 'background: #f8f9fa; padding: 20px; margin-bottom: 15px; border-radius: 8px;';
                matchDiv.innerHTML = `
                    <h3>Match ${i + 1}</h3>
                    <div style="display: grid; grid-template-columns: 1fr 1fr 1fr 1fr 1fr; gap: 15px;">
                        <div class="form-group">
                            <label>Team 1</label>
                            <select name="pairings[${i}][team1]" class="form-control" required onchange="updateTeamDropdowns()">
                                <option value="">Select Team 1</option>
                                <?php foreach ($all_teams as $team): ?>
                                <option value="<?php echo $team['id']; ?>">
                                    <?php echo htmlspecialchars($team['name']); ?> (Pool <?php echo $team['pool_number']; ?>)
                                </option>
                                <?php endforeach; ?>
                            </select>
                        </div>
                        
                        <div class="form-group">
                            <label>Team 2</label>
                            <select name="pairings[${i}][team2]" class="form-control" required onchange="updateTeamDropdowns()">
                                <option value="">Select Team 2</option>
                                <?php foreach ($all_teams as $team): ?>
                                <option value="<?php echo $team['id']; ?>">
                                    <?php echo htmlspecialchars($team['name']); ?> (Pool <?php echo $team['pool_number']; ?>)
                                </option>
                                <?php endforeach; ?>
                            </select>
                        </div>
                        
                        <div class="form-group">
                            <label>Referee</label>
                            <select name="pairings[${i}][referee]" class="form-control">
                                <option value="">No Referee</option>
                                <?php 
                                $referees_result->data_seek(0);
                                while ($ref = $referees_result->fetch_assoc()): 
                                ?>
                                <option value="<?php echo $ref['id']; ?>"><?php echo htmlspecialchars($ref['name']); ?></option>
                                <?php endwhile; ?>
                            </select>
                        </div>
                        
                        <div class="form-group">
                            <label>Court</label>
                            <input type="text" name="pairings[${i}][court]" class="form-control" placeholder="e.g., Court 1">
                        </div>
                        
                        <div class="form-group">
                            <label>Time</label>
                            <input type="text" name="pairings[${i}][time]" class="form-control" placeholder="e.g., 09:00 AM">
                        </div>
                    </div>
                `;
                container.appendChild(matchDiv);
            }
        }
    </script>
</head>
<body>
    <div class="nav-bar">
        <div class="container">
            <a href="admin.php">Dashboard</a>
            <a href="admin_setup.php">Tournament Setup</a>
            <a href="admin_matches.php">Generate Matches</a>
            <a href="admin_manage_matches.php">Manage Matches</a>
            <a href="admin_knockout.php">Knockout Setup</a>
            <a href="logout.php">Logout</a>
        </div>
    </div>
    
    <div class="container">
        <h1>Knockout Stage Setup</h1>
        
        <?php if ($message): ?>
            <div class="alert alert-success"><?php echo $message; ?></div>
        <?php endif; ?>
        
        <?php if ($knockout_count == 0): ?>
        <div class="form-card">
            <h2>Create Knockout Bracket</h2>
            <p><strong>Note:</strong> After you create the initial knockout matches, all future rounds will automatically advance winners when matches are completed.</p>
            
            <div class="form-group">
                <label>Number of Teams in Knockout Stage</label>
                <select id="num_teams" class="form-control" onchange="calculateMatches()" style="max-width: 300px;">
                    <option value="">Select number of teams</option>
                    <option value="64">64 Teams (Round of 64)</option>
                    <option value="32">32 Teams (Round of 32)</option>
                    <option value="16">16 Teams (Round of 16)</option>
                    <option value="8">8 Teams (Quarter-Finals)</option>
                    <option value="4">4 Teams (Semi-Finals)</option>
                    <option value="2">2 Teams (Finals)</option>
                </select>
            </div>
            
            <form method="POST" id="knockoutForm">
                <div id="matchesContainer"></div>
                
                <div id="submitButton" style="display: none; margin-top: 20px;">
                    <button type="submit" name="create_initial_knockout" class="btn btn-success">
                        Create Knockout Bracket
                    </button>
                </div>
            </form>
            
            <script>
                document.getElementById('num_teams').addEventListener('change', function() {
                    if (this.value) {
                        document.getElementById('submitButton').style.display = 'block';
                    } else {
                        document.getElementById('submitButton').style.display = 'none';
                    }
                });
            </script>
        </div>
        <?php endif; ?>
        
        <?php if (!empty($rounds_display)): ?>
        <div class="form-card">
            <h2>Knockout Matches Overview</h2>
            <p style="color: #27ae60; font-weight: bold; margin-bottom: 15px;">Rounds automatically advance when all matches are completed</p>
            <?php foreach ($rounds_display as $round_num => $matches): 
                $round_label = getRoundNameByMatches(count($matches));
            ?>
            <h3>Round <?php echo $round_num; ?>: <?php echo $round_label; ?></h3>
            <div class="match-table">
                <table>
                    <thead>
                        <tr>
                            <th>Match</th>
                            <th>Court</th>
                            <th>Time</th>
                            <th>Team 1</th>
                            <th>Score</th>
                            <th>Team 2</th>
                            <th>Score</th>
                            <th>Referee</th>
                            <th>Status</th>
                        </tr>
                    </thead>
                    <tbody>
                        <?php 
                        $match_num = 1;
                        foreach ($matches as $match): 
                        ?>
                        <tr>
                            <td><strong><?php echo $match_num++; ?></strong></td>
                            <td><?php echo htmlspecialchars($match['court_name'] ?? 'TBA'); ?></td>
                            <td><?php echo htmlspecialchars($match['start_time'] ?? 'TBA'); ?></td>
                            <td><?php echo htmlspecialchars($match['team1_name']); ?></td>
                            <td><strong><?php echo $match['team1_score']; ?></strong></td>
                            <td><?php echo htmlspecialchars($match['team2_name']); ?></td>
                            <td><strong><?php echo $match['team2_score']; ?></strong></td>
                            <td><?php echo htmlspecialchars($match['referee_name'] ?? 'N/A'); ?></td>
                            <td>
                                <span class="status-badge status-<?php echo $match['status']; ?>">
                                    <?php echo str_replace('_', ' ', ucfirst($match['status'])); ?>
                                </span>
                            </td>
                        </tr>
                        <?php endforeach; ?>
                    </tbody>
                </table>
            </div>
            <?php endforeach; ?>
        </div>
        <?php endif; ?>